//==============================================================================
// Project:		Wealth Transfers and their economic effects
// File name:	Crawford Hood comparison
// Objective: 	Generate figure B.9, panel b, and figure 2.6/table B.7 statistic that is compared with Crawford and Hood's research
//
//==============================================================================

clear all

************************
*** Data import ***
************************

cd "XXXX" //set content directory here

use combined_filtered_restricted.dta, replace

merge m:1 xwaveid wave using bequest_from_HH_death.dta //flag for inheritances due to HH deaths
drop if _merge == 2
drop _merge

merge m:1 wave using CPI.dta //CPI data
drop if _merge == 2
drop _merge 

xtset xwaveid wave

drop if hgint == 0 //drop non-responders

************************
*** Create variables ***
************************

gen age_range = 1 if hgage <= wave + 61 & hgage >= wave + 47 //people not aged 65-79 not considered part of the HH for the per-person calculation purposes
keep if age_range == 1

replace oifinha = 0 if oifinha == -1 //inheritances
replace oifinha = . if oifinha < 0
gen bequest = oifinha
replace bequest = 0 if bequest > 0 & bequest != . & bequest_from_HH_death == 1 //filter out inheritances due to first estates

replace oifnptr = . if oifnptr < 0 //gifts
gen gift = oifnptr + oifnpt

gen return = 1 //three percent real return to wealth transfers assumed as per Crawford and Hood (p. 62)
local returns 18 17 16 15 14 13 12 11 10 9 8 7 6 5 4 3 2 1 
foreach x of local returns{
replace return = return*1.03 if wave <= `x' 
}

replace hwnwip = hwnwip * 100 * return / CPI
replace hwnwin = hwnwin * 100 * return / CPI
replace bequest = bequest * 100 * return / CPI
replace gift = gift * 100 * return / CPI

gen transfer_1 = bequest + gift
forvalues x = 2/18{
	local y = `x' - 1
	gen transfer_`x' = transfer_`y' + L`y'.bequest + L`y'.gift  
}

******************************************
*** Create sample and scaled variables ***
******************************************

drop if hgage > 79
drop if hgage < 65 //people not aged 65-79 not considered part of the HH for the per-person calculation purposes (see email from Rowena Crawford on 10/05/21), so remove them prior to determining HH size.
drop if transfer_18 == .
drop if wlra_r == 0
drop if wave != 18


gen flag = 1
egen HH_members = total(flag), by(hhrhid) //HH members aged 65-79
gen net_worth_plus_transfer_pp = (hwnwip - hwnwin) / HH_members

forvalues x = 1/18{
egen transfer_`x'_HH = total(transfer_`x'), by(hhrhid)
gen transfer_`x'_pp = transfer_`x'_HH / HH_members
gen net_worth_pp_`x' = (hwnwip - hwnwin - transfer_`x'_HH) / HH_members
}


************************
*** Calculate Ginis ****
************************

forvalues x = 1/18{
sgini net_worth_pp_`x' transfer_`x'_pp [pweight = wlra_r], source
matrix Gini_`x' = r(coeffs)
matrix Corr_`x' = r(r)
matrix Share_`x' = r(s)
}

clear

forvalues x = 1/18{
svmat Gini_`x', names(col)
rename net_worth_pp_`x' Gini_net_worth_`x'
rename transfer_`x'_pp Gini_transfer_pp_`x'
drop _factotal
svmat Corr_`x', names(col)
rename net_worth_pp_`x' Corr_net_worth_`x'
rename transfer_`x'_pp Corr_transfer_pp_`x'
svmat Share_`x', names(col)
rename net_worth_pp_`x' Share_net_worth_`x'
rename transfer_`x'_pp Share_transfer_pp_`x'
}

gen flag = 1

reshape long Gini_net_worth_ Gini_transfer_pp_  Corr_net_worth_ Corr_transfer_pp_  Share_net_worth_ Share_transfer_pp_ , i(flag) j(years)

gen effect = ((Share_transfer_pp_ * Corr_transfer_pp_ * Gini_transfer_pp_ + Share_net_worth_ * Corr_net_worth_ * Gini_net_worth_) - Gini_net_worth_) * 100 / Gini_net_worth_

gen neg_effect = - effect

export delim Crawford_Hood_comparison.csv, replace //figure B.9, panel b; wave 18 value is reported in table B.7 and figure 2.6









